<?php
namespace Bookly\Lib\Cloud;

use Bookly\Lib\Utils;
use Exception;

/**
 * Class SMS
 * @package Bookly\Lib\Cloud
 */
class SMS extends Base
{
	
    const CANCEL_SENDER_ID    = '/1.0/users/%token%/sender-ids/cancel'; //GET
    const CHANGE_SMS_STATUS   = '/1.0/users/%token%/sms';               //PATCH
    const GET_PRICES          = '/1.0/prices';                          //GET
    const GET_SENDER_IDS_LIST = '/1.0/users/%token%/sender-ids';        //GET
    const GET_SMS_LIST        = '/1.0/users/%token%/sms';               //GET
    const REQUEST_SENDER_ID   = '/1.0/users/%token%/sender-ids';        //POST
    const RESET_SENDER_ID     = '/1.0/users/%token%/sender-ids/reset';  //GET
    const SEND_SMS            = '/1.1/users/%token%/sms';               //POST

    /** @var array */
    protected $sender_id;

    /**
     * convert persian number to english
     * 
     * @param string $string
     * 
     * @return string
     */
    public function convert($string)
    {
        $persian = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
        $arabic = ['٩', '٨', '٧', '٦', '٥', '٤', '٣', '٢', '١', '٠'];

        $num = range(0, 9);
        $convertedPersianNums = str_replace($persian, $num, $string);
        $englishNumbersOnly = str_replace($arabic, $num, $convertedPersianNums);

        return $englishNumbersOnly;
    }

    /**
     * Send SMS.
     *
     * @param string $phone_number
     * @param string $message
     * @param string $impersonal_message
     * @param int    $type_id
     * @return bool
     */
    public function sendSms($phone_number, $message, $impersonal_message, $type_id = null, $meliBodyID = null)
    {
        $response = new  \stdClass;
            $data = array(
                'message'            => $message,
                'impersonal_message' => $impersonal_message,
                'phone'              => $this->normalizePhoneNumber( $phone_number ),
                'type'               => $type_id,
            );

        $to = $this->convert($this->normalizePhoneNumber($phone_number));
            if ( $data['phone'] != '' ) {

            $username = get_option('bookly_sms_melipayamak_username');
            $password = get_option('bookly_sms_melipayamak_password');
            $bookly_sms_melipayamak_share_service_url = 'https://rest.payamak-panel.com/api/SendSMS/BaseServiceNumber';
            $text = $message;
            $to = $this->convert($this->normalizePhoneNumber($phone_number));

            if (get_option('bookly_sms_melipayamak_enabled') == "1") {

                if ($bookly_sms_melipayamak_share_service_url && $meliBodyID != null && $meliBodyID != 0 && $meliBodyID != '') {

                    $bodyId = $meliBodyID;
                    $curl = curl_init();
                    try {
                        curl_setopt_array($curl, array(
                            CURLOPT_URL => $bookly_sms_melipayamak_share_service_url,
                            CURLOPT_RETURNTRANSFER => true,
                            CURLOPT_ENCODING => '',
                            CURLOPT_MAXREDIRS => 10,
                            CURLOPT_TIMEOUT => 0,
                            CURLOPT_FOLLOWLOCATION => true,
                            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                            CURLOPT_CUSTOMREQUEST => 'POST',
                            CURLOPT_POSTFIELDS => "username={$username}&password={$password}&text={$text}&to={$to}&bodyId={$bodyId}",
                            CURLOPT_HTTPHEADER => array(
                                'Content-Type: application/x-www-form-urlencoded'
                            ),
                        ));

                        $responseApi = curl_exec($curl);

                        curl_close($curl);

                        $responseApiDecode = json_decode($responseApi);

                        $response->gateway_status = $responseApiDecode->RetStatus == 1 ? 1 : -1;
                    } catch (\Exception $e) {
                    }
                } else {
                    $from = get_option('bookly_sms_melipayamak_panel_number');
                    try {
                        $curl = curl_init();

                        curl_setopt_array(
                            $curl,
                            array(
                                CURLOPT_URL => 'https://rest.payamak-panel.com/api/SendSMS/SendSMS',
                                CURLOPT_RETURNTRANSFER => true,
                                CURLOPT_ENCODING => '',
                                CURLOPT_MAXREDIRS => 10,
                                CURLOPT_TIMEOUT => 0,
                                CURLOPT_FOLLOWLOCATION => true,
                                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                                CURLOPT_CUSTOMREQUEST => 'POST',
                                CURLOPT_POSTFIELDS => "username=$username&password=$password&to=$to&from=$from&text=$text",
                                CURLOPT_HTTPHEADER => array(
                                    'Content-Type: application/x-www-form-urlencoded'
                                ),
                            )
                        );

                        $res = curl_exec($curl);
                        curl_close($curl);
                        $result = json_decode($res);

                        if (isset($result->RetStatus) && $result->RetStatus == 1) {
                            $response->gateway_status = 1;
                        } else {
                            $response->gateway_status = -1;
                        }
                    } catch (\Throwable $th) {
                        $response->gateway_status = -1;
                    }
                }
            }

            if (get_option('bookly_sms_ippanel_enabled') == "1") {
                $baseURL = strtolower(trim(get_option('bookly_sms_ippanel_url')));
                if (empty($baseURL)) $baseURL = "ippanel.com";
                $user = trim(get_option('bookly_sms_ippanel_username'));
                $pass = trim(get_option('bookly_sms_ippanel_password'));
                $from = trim(get_option('bookly_sms_ippanel_panel_number'));
                $patternId = $meliBodyID;
                $message  = str_replace(array('https://', 'http://', 'www.'), array('', '', ''), $message);

                if ($phone_number != '') {
                    if (!is_array($phone_number)) $phone_number = explode(',', $this->convert($this->normalizePhoneNumber($phone_number)));
                    if ($meliBodyID != null && $meliBodyID != '') {
                        try {
                            $message = str_replace("\r\n", ';', $message);
                            $message = str_replace("\n", ';', $message);

                            $splited = explode(';', $message);
                            $resArray = array();
                            foreach ($splited as $parm) {
                                $splited_parm = explode('=', $parm);
                                $resArray[trim($splited_parm[0])] = trim($splited_parm[1]);
                            }

                            $input_data = $resArray;

                            foreach ($phone_number as $to) {
                                $toNum = array($to);
                                $url = "/patterns/pattern?username=" . $user . "&password=" . urlencode($pass) . "&from=" . $from . "&to=" . json_encode($toNum) . "&input_data=" . urlencode(json_encode($input_data)) . "&pattern_code=" . $patternId;
                                $handler = curl_init($baseURL . $url);
                                curl_setopt($handler, CURLOPT_CONNECTTIMEOUT, 5);
                                curl_setopt($handler, CURLOPT_TIMEOUT, 20);
                                curl_setopt($handler, CURLOPT_CUSTOMREQUEST, 'GET');
                                curl_setopt($handler, CURLOPT_RETURNTRANSFER, true);
                                $res = curl_exec($handler);
                                $result = json_decode($res);
                                curl_close($handler);
                                if (is_array($result) && $result[0] != 'sent') {
                                    $res_code = $result[0];
                                    $res_data = $result[1];
                                    error_log('Bookly -> SMS.php panel-response: ' . $res);
                                    error_log('Bookly -> SMS.php error: ' . $this->getPanelErrors($res_code));
                                    $response->gateway_status = 0;
                                } else {
                                    $response->gateway_status = 1;
                                }
                            }
                        } catch (\Exception $e) {
                            error_log('Bookly -> SMS.php error Pattern IPPanel: ' . json_encode($e));
                            $response->gateway_status = 0;
                        }
                    } else {
                        try {
                            $params = array(
                                'uname'        =>    $user,
                                'pass'        =>    $pass,
                                'from'        =>    $from,
                                'message'    =>    $message,
                                'to'        =>    json_encode($phone_number),
                                'op'        =>    'send',
                            );
                            $url = "/services.jspd";
                            $handler = curl_init($baseURL . $url);
                            curl_setopt($handler, CURLOPT_CONNECTTIMEOUT, 5);
                            curl_setopt($handler, CURLOPT_TIMEOUT, 20);
                            curl_setopt($handler, CURLOPT_CUSTOMREQUEST, 'POST');
                            curl_setopt($handler, CURLOPT_POSTFIELDS, $params);
                            curl_setopt($handler, CURLOPT_RETURNTRANSFER, true);
                            $res = curl_exec($handler);
                            $result = json_decode($res);
                            curl_close($handler);
                            $res_code = $result[0];
                            $res_data = $result[1];
                            if ($res_code == 0 && !is_null($res_code)) {
                                $response->gateway_status = 1;
                            } else {
                                error_log('Bookly -> SMS.php error: ' . $this->getPanelErrors($res_code));
                                $response->gateway_status = 0;
                            }
                        } catch (\Exception $e) {
                            error_log('Bookly -> SMS.php error Normal IPPanel: ' . json_encode($e));
                            $response->gateway_status = 0;
                        }
                    }
                } else {
                    $this->errors[] = __('Phone number is empty.', 'bookly');
                    $response->gateway_status = 0;
                }
            }

            if (get_option('bookly_sms_parsgreen_enabled') == "1") {
                try {
				$url = 'http://sms.parsgreen.ir/Apiv2/Message/SendSms';
                    
                    $ch = curl_init($url);
                    $jsonDataEncoded = json_encode(['SmsBody' => $text, 'Mobiles' => [$to], 'SmsNumber' => get_option('bookly_sms_parsgreen_panel_number')]);
                    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonDataEncoded);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
                    $header = array('authorization: BASIC APIKEY:' . get_option('bookly_sms_parsgreen_apikey'), 'Content-Type: application/json;charset=utf-8');
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
                    $result = curl_exec($ch);
                    $res = json_decode($result);
                    curl_close($ch);

                    if ($res->R_Success) {
                        $response->gateway_status = 1;
                    } else {
                        error_log('Bookly -> SMS.php error: ' . $this->getPanelErrors($res_code));
                        $response->gateway_status = 0;
                    }
                } catch (\Exception $e) {
                    error_log('Bookly -> SMS.php error Normal IPPanel: ' . json_encode($e));
                    $response->gateway_status = 0;
                }
            }

            if (get_option('bookly_sms_kavehnegar_enabled') == "1") {

                try {
                    $url = 'http://api.kavenegar.com/v1/'. get_option('bookly_sms_kavehnegar_apikey') .'/sms/send.json';
                    $fields_string = http_build_query([
                        'receptor' => $to,
                        'sender' => get_option('bookly_sms_kavehnegar_panel_number'),
                        'message' => $text
                    ]);

                    $curl = curl_init();

                    curl_setopt_array($curl, array(
                        CURLOPT_URL => $url,
                        CURLOPT_RETURNTRANSFER => true,
                        CURLOPT_ENCODING => '',
                        CURLOPT_MAXREDIRS => 10,
                        CURLOPT_TIMEOUT => 0,
                        CURLOPT_FOLLOWLOCATION => true,
                        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                        CURLOPT_CUSTOMREQUEST => 'POST',
                        CURLOPT_POSTFIELDS => $fields_string,
                        CURLOPT_HTTPHEADER => array(
                            'Content-Type: application/x-www-form-urlencoded'
                        ),
                    ));

                    $result = curl_exec($curl);
                    $res = json_decode($result);
                    curl_close($curl);
                    
                    if ($res->return->status == 200) {
                        $response->gateway_status = 1;
                    } else {
                        error_log('Bookly -> SMS.php error: ' . $res->return->message);
                        $response->gateway_status = 0;
                    }
                } catch (\Exception $e) {
                    error_log('Bookly -> SMS.php error Normal KavehNegar: ' . json_encode($e));
                    $response->gateway_status = 0;
                }
                    }

            if (get_option('bookly_sms_tsms_enabled') == "1") {

                try {
                    
                    $tsmsPanelNumber = get_option('bookly_sms_tsms_panel_number');
                    $username = get_option('bookly_sms_tsms_username');
                    $password = get_option('bookly_sms_tsms_password');
                    $tsmsUrlSend = "https://tsms.ir/url/tsmshttp.php?from={$tsmsPanelNumber}&to={$to}&username={$username}&password={$password}&message=" . urlencode($text);

                    $curl = curl_init();

                    curl_setopt_array($curl, array(
                    CURLOPT_URL => $tsmsUrlSend,
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_ENCODING => '',
                    CURLOPT_MAXREDIRS => 10,
                    CURLOPT_TIMEOUT => 0,
                    CURLOPT_FOLLOWLOCATION => true,
                    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                    CURLOPT_CUSTOMREQUEST => 'GET',
                    CURLOPT_HTTPHEADER => array(
                        'Connection: keep-alive',
                        'Cache-Control: max-age=0',
                        'sec-ch-ua-mobile: ?0',
                        'sec-ch-ua-platform: "Windows"',
                        'Upgrade-Insecure-Requests: 1',
                        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.9',
                        'Sec-Fetch-Site: none',
                        'Sec-Fetch-Mode: navigate',
                        'Sec-Fetch-User: ?1',
                        'Sec-Fetch-Dest: document',
                        'Accept-Language: en-US,en;q=0.9'
                    ),
                    ));

                    $responseCurl = curl_exec($curl);

                    curl_close($curl);
                    
                    if ($responseCurl > 0) {
                        $response->gateway_status = 1;
                    } else {
                        error_log('Bookly -> SMS.php error: ' . $res->return->message);
                        $response->gateway_status = 0;
                    }
                } catch (Exception $e) {
                    error_log('Bookly -> SMS.php error Normal Tsms: ' . json_encode($e));
                    $response->gateway_status = 0;
                }
            }

            if (get_option('bookly_sms_sahandsms_enabled') == "1") {

                try {

                    $username = get_option('bookly_sms_sahandsms_username');
                    $password = get_option('bookly_sms_sahandsms_password');
                    $from = get_option('bookly_sms_sahandsms_panel_number');

                    $getFields = http_build_query([
                        'username' => $username,
                        'password' => $password,
                        'message' => $message,
                        'fromNumber' => $from,
                        'toNumbers' => $phone_number
                    ]);

                    $url = "http://webservice.sahandsms.com/NewSMSWebService.asmx/Send?{$getFields}";

                    $curl = curl_init();

                    curl_setopt_array($curl, array(
                        CURLOPT_URL => $url,
                        CURLOPT_RETURNTRANSFER => true,
                        CURLOPT_ENCODING => '',
                        CURLOPT_MAXREDIRS => 10,
                        CURLOPT_TIMEOUT => 0,
                        CURLOPT_FOLLOWLOCATION => true,
                        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                        CURLOPT_CUSTOMREQUEST => 'GET',
                    ));

                    $result = curl_exec($curl);
                    $httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

                    curl_close($curl);

                    if ($httpcode == 200) {
                        $response->gateway_status = 1;
                        } else {
                        error_log('Bookly -> sahandSms SMS.php error: ' . $res->return->message);
                        $response->gateway_status = 0;
                        }
                } catch (\Exception $e) {
                    error_log('Bookly -> SMS.php error Normal sahandSms: ' . json_encode($e));
                    $response->gateway_status = 0;
                    }
                }

            if (get_option('bookly_sms_hamyarsms_enabled') == "1") {

                try {

                    ini_set("soap.wsdl_cache_enabled", "0");
                    $sms_client = new \SoapClient('http://payamakapi.ir/SendService.svc?wsdl', array('encoding'=>'UTF-8'));

                    $username = get_option('bookly_sms_hamyarsms_username');
                    $password = get_option('bookly_sms_hamyarsms_password');
                    $from = get_option('bookly_sms_hamyarsms_panel_number');

                    try {
                        $parameters['userName'] = $username;
                        $parameters['password'] = $password;
                        $parameters['fromNumber'] = $from;
                        $parameters['toNumbers'] = array( $phone_number );
                        $parameters['messageContent'] = $message;
                        $parameters['isFlash'] = false;
                        $recId = array(0);
                        $status = 0x0;
                        $parameters['recId'] = &$recId ;
                        $parameters['status'] = &$status ;
                        $statusend =  $sms_client->SendSMS($parameters)->SendSMSResult;
                    }  
                    catch (\Exception $e) {
                        error_log('Bookly -> SMS.php error Normal sahandSms: ' . json_encode($e));
                        $response->gateway_status = 0;
                    }
                    finally {

                        if (1 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: نام کاربری یا کلمه عبور درست نمی باشد :');
                            $response->gateway_status = 0;
                        }
                        if (2 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: کاربر مسدود شده است : ');
                            $response->gateway_status = 0;
                        }
                        if (3 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: شماره فرستنده نامعتبر است : ');
                            $response->gateway_status = 0;
                        }
                        if (4 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: محدودیت در ارسال روزانه : ');
                            $response->gateway_status = 0;
                        }
                        if (5 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: تعداد گیرندگان حداکثر 100 شماره می باشد :');
                            $response->gateway_status = 0;
                        }
                        if (6 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: خط فرسنتده غیرفعال است : ');
                            $response->gateway_status = 0;
                        }
                        if (7 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: متن پیامک شامل کلمات فیلتر شده است : ');
                            $response->gateway_status = 0;
                        }
                        if (8 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: اعتبار کافی نیست : ');
                            $response->gateway_status = 0;
                        }
                        if (9 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: سامانه در حال بروز رسانی می باشد : ');
                            $response->gateway_status = 0;
                        }
                        if (10 == $statusend) {
                            error_log('Bookly -> hamyarsms SMS.php error: وب سرویس غیرفعال است : ');
                            $response->gateway_status = 0;
                        }

                        $response->gateway_status = 1;
                    }
                } catch (\Exception $e) {
                    error_log('Bookly -> SMS.php error Normal sahandSms: ' . json_encode($e));
                    $response->gateway_status = 0;
                }
            }

            if (get_option('bookly_sms_smsir_enabled') == "1") {
                try {
                    $postData = array(
                        'UserApiKey' => get_option('bookly_sms_smsir_secret_key'),
                        'SecretKey' => get_option('bookly_sms_smsir_api_key'),
                        'System' => 'php_rest_v_2_0'
                    );
                    $postString = json_encode($postData);
            
                    $ch = curl_init("https://ws.sms.ir/api/Token");
                    curl_setopt(
                        $ch, CURLOPT_HTTPHEADER, array(
                            'Content-Type: application/json'
                        )
                    );
                    curl_setopt($ch, CURLOPT_HEADER, false);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $postString);
            
                    $result = curl_exec($ch);
                    curl_close($ch);
            
                    $resp= json_decode($result);
            
                    $token = false;
                    $IsSuccessful = '';
                    $TokenKey = '';
                    if (is_object($resp)) {
                        $IsSuccessful = $resp->IsSuccessful;
                        if ($IsSuccessful == true) {
                            $TokenKey = $resp->TokenKey;
                            $token = $TokenKey;
                        } else {
                            $token = false;
                        }
                    }

                    if ($token != false) {
                        $postData = array(
                            'Messages' => array($message),
                            'MobileNumbers' => array($to),
                            'LineNumber' => get_option('bookly_sms_smsir_number'),
                            'SendDateTime' => date("Y-m-d")."T".date("H:i:s"),
                            'CanContinueInCaseOfError' => 'false'
                        );
            
                        
                        $postString = json_encode($postData);

                        $ch = curl_init('https://ws.sms.ir/api/MessageSend');
                        curl_setopt(
                            $ch, CURLOPT_HTTPHEADER, array(
                                'Content-Type: application/json',
                                'x-sms-ir-secure-token: '.$token
                            )
                        );
                        curl_setopt($ch, CURLOPT_HEADER, false);
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                        curl_setopt($ch, CURLOPT_POSTFIELDS, $postString);

                        $result = curl_exec($ch);
                        curl_close($ch);


                        $object = json_decode($result);
            
                        $result = false;
                        if (is_object($object)) {
                            $response->gateway_status = 1;
                            $result = $object->Message;
                        } else {
                            $response->gateway_status = -1;
                            $result = false;
                        }
                    } else {
                        $response->gateway_status = -1;
                        $result = false;
                    }


                } catch (\Exception $e) {
                    $response->gateway_status = -1;
                }
            }
            
            } else {
            $this->errors[] = __('Phone number is empty.', 'bookly');
            }

        if (isset($response->gateway_status) && $response->gateway_status == 1) {
            return true;
        }

        return false;
    }

    private function getPanelErrors($error)
    {
        $errorCodes = array(
            '0'  => 'عمليات با موفقيت انجام شده است.',
            '1'  => 'متن پيام خالي مي باشد.',
            '2'  => 'کاربر محدود گرديده است.',
            '3'  => 'خط به شما تعلق ندارد.',
            '4'  => 'گيرندگان خالي است.',
            '5'  => 'اعتبار کافي نيست.',
            '7'  => 'خط مورد نظر براي ارسال انبوه مناسب نميباشد.',
            '9'  => 'خط مورد نظر در اين ساعت امکان ارسال ندارد.',
            '98' => 'حداکثر تعداد گيرنده رعايت نشدهه است.',
            '99' => 'اپراتور خط ارسالي قطع مي باشد.',
            '21' => 'پسوند فايل صوتي نامعتبر است.',
            '22' => 'سايز فايل صوتي نامعتبر است.',
            '23' => 'تعداد تالش در پيام صوتي نامعتبر است.',
            '100' => 'شماره مخاطب دفترچه تلفن نامعتبر مي باشد.',
            '101' => 'شماره مخاطب در دفترچه تلفن وجود دارد.',
            '102' => 'شماره مخاطب با موفقيت در دفترچه تلفن ذخيره گرديد.',
            '111' => 'حداکثر تعداد گيرنده براي ارسال پيام صوتي رعايت نشده است.',
            '131' => 'تعداد تالش در پيام صوتي بايد يکبار باشد.',
            '132' => 'آدرس فايل صوتي وارد نگرديده است.',
            '301' => 'از حرف ويژه در نام کاربري استفاده گرديده است.',
            '302' => 'قيمت گذاري انجام نگريدهه است.',
            '303' => 'نام کاربري وارد نگرديده است.',
            '304' => 'نام کاربري قبال انتخاب گرديده است.',
            '305' => 'نام کاربري وارد نگرديده است.',
            '306' => 'کد ملي وارد نگرديده است.',
            '307' => 'کد ملي به خطا وارد شده است.',
            '308' => 'شماره شناسنامه نا معتبر است.',
            '309' => 'شماره شناسنامه وارد نگرديده است.',
            '310' => 'ايميل کاربر وارد نگرديده است.',
            '311' => 'شماره تلفن وارد نگرديده است.',
            '312' => 'تلفن به درستي وارد نگرديده است.',
            '313' => 'آدرس شما وارد نگرديده است.',
            '314' => 'شماره موبايل را وارد نکرده ايد.',
            '315' => 'شماره موبايل به نادرستي وارد گرديده است.',
            '316' => 'سطح دسترسي به نادرستي وارد گرديده است.',
            '317' => 'کلمه عبور وارد نگرديده است.',
            '404' => 'پترن در دسترس نيست.',
            '455' => 'ارسال در آينده براي کد بالک ارسالي لغو شد.',
            '456' => 'کد بالک ارسالي نامعتبر است.',
            '458' => 'کد تيکت نامعتبر است.',
            '964' => 'شما دسترسي نمايندگي نداريد.',
            '962' => 'نام کاربري يا کلمه عبور نادرست مي باشد.',
            '963' => 'دسترسي نامعتبر مي باشد.',
            '971' => 'پترن ارسالي نامعتبر است.',
            '970' => 'پارامتر هاي ارسالي براي پترن نامعتبر است.',
            '972' => 'دريافت کننده براي ارسال پترن نامعتبر مي باشد.',
            '992' => 'ارسال پيام از ساعت 8 تا 23 مي باشد.',
            '993' => 'دفترچه تلفن بايد يک آرايه باشد',
            '994' => 'لطفا تصويري از کارت بانکي خود را از منو مدارک ارسال کنيد',
            '995' => 'جهت ارسال با خطوط اشتراکي سامانه، لطفا شماره کارت بانکيه خود را به دليل تکميل فرايند  احراز هويت از بخش ارسال مدارک ثبت نماييد.',
            '996' => 'پترن فعال نيست.',
            '997' => 'شما اجازه ارسال از اين پترن را نداريد.ه',
            '998' => 'کارت ملي يا کارت بانکي شما تاييد نشده است.',
            '1001' => 'فرمت نام کاربري درست نمي باشد)حداقله ? کاراکتر، فقط حروف و اعداد(.',
            '1002' => 'گذر واژه خيلي ساده مي باشد)حداقل ? کاراکتر بوده و نام کاربري،',
            'ايمي' => ' و شماره موبايل در آن وجود نداشته باشد(.',
            '1003' => 'مشکل در ثبت، با پشتيباني تماس بگيريد.',
            '1004' => 'مشکل در ثبت، با پشتيباني تماس بگيريد.',
            '1005' => 'مشکل در ثبت، با پشتيباني تماس بگيريد.',
            '1006' => 'تاريخ ارسال پيام براي گذشته مي باشد، لطفا تاريخ ارسال پيام را به درستي وارد نماييد.ه',
        );
        return isset($errorCodes[$error]) ? $errorCodes[$error] : 'اشکال تعريف نشده با کد :' . $error;
    }

    /**
     * Change SMS product status.
     *
     * @param bool $status
     * @return bool
     */
    public function changeSmsStatus( $status )
    {
        $data = array(
            'status' => $status
        );

        $response = $this->api->sendPatchRequest(self::CHANGE_SMS_STATUS, $data);
        if ( $response ) {
            update_option( 'bookly_cloud_account_products', $response['products'] );

            return true;
        }
        return false;
    }

    /**
     * Return phone_number in international format without +
     *
     * @param $phone_number
     * @return string
     */
    public function normalizePhoneNumber( $phone_number )
    {
        // Remove everything except numbers and "+".
        $phone_number = $this->convert($phone_number);
        $phone_number = preg_replace( '/[^\d\+]/', '', $phone_number );

        if ( strpos( $phone_number, '+' ) === 0 ) {
            // ok.
        } elseif ( strpos( $phone_number, '00' ) === 0 ) {
	            $phone_number = ltrim( $phone_number, '0' );
	        } else {
            // Default country code can contain not permitted characters. Remove everything except numbers.

	            $phone_number = ltrim( preg_replace( '/\D/', '', get_option( 'bookly_cst_default_country_code', '' ) ), '0' )  . ltrim( $phone_number, '0' );
			}

        // Finally remove "+" if there were any among digits.
        return str_replace( '+', '', $phone_number );
    }

    /**
     * Get SMS list.
     *
     * @param null $start_date
     * @param null $end_date
     * @return array
     */
    public function getSmsList( $start_date = null, $end_date = null )
    {
        if ( $this->api->getToken() ) {
            $response = $this->api->sendGetRequest(
                self::GET_SMS_LIST,
				 compact( 'start_date', 'end_date' )
				);
            if ( $response ) {
                array_walk( $response['list'], function( &$item ) {
                    $date_time = Utils\DateTime::UTCToWPTimeZone( $item['datetime'] );
                    $item['date']    = Utils\DateTime::formatDate( $date_time );
                    $item['time']    = Utils\DateTime::formatTime( $date_time );
                    $item['message'] = nl2br( preg_replace( '/([^\s]{50})+/U', '$1 ', htmlspecialchars( $item['message'] ) ) );
                    $item['phone']   = '+' . $item['phone'];
                    $item['charge'] = $item['charge'] === null ? '' : rtrim( $item['charge'], '0' );
                    $item['info'] = $item['info'] === null ? '' : nl2br( htmlspecialchars( $item['info'] ) );
                    switch ( $item['status'] ) {
                        case 1:
                        case 10:
                            $item['status'] = __( 'Queued', 'bookly' );
                            $item['charge'] = '$' . $item['charge'];
                            break;
                        case 2:
                        case 16:
                            $item['status'] = __( 'Error', 'bookly' );
                            $item['charge'] = '';
                            break;
                        case 3:
                            $item['status'] = __( 'Out of credit', 'bookly' );
                            $item['charge'] = '';
                            break;
                        case 4:
                            $item['status'] = __( 'Country out of service', 'bookly' );
                            $item['charge'] = '';
                            break;
                        case 5:
                            $item['status'] = __( 'Blocked', 'bookly' );
                            $item['charge'] = '';
                            break;
                        case 11:
                            $item['status'] = __( 'Sending', 'bookly' );
                            $item['charge'] = '$' . $item['charge'];
                            break;
                        case 12:
                            $item['status'] = __( 'Sent', 'bookly' );
                            $item['charge'] = '$' . $item['charge'];
                            break;
                        case 13:
                            $item['status'] = __( 'Delivered', 'bookly' );
                            $item['charge'] = '$' . $item['charge'];
                            break;
                        case 14:
                            $item['status'] = __( 'Failed', 'bookly' );
                            if ( $item['charge'] != '' ) {
                                $item['charge'] = '$' . $item['charge'];
                            }
                            break;
                        case 15:
                            $item['status'] = __( 'Undelivered', 'bookly' );
                            $item['charge'] = '$' . $item['charge'];
                            break;
                        default:
                            $item['status'] = __( 'Error', 'bookly' );
                            $item['charge'] = '';
                    }
                } );

                $this->setUndeliveredSmsCount( 0 );

                return $response;
            }
        }

        return array( 'success' => false, 'list' => array() );
    }

    /**
     * Get Price list.
     *
     * @return array
     */
    public function getPriceList()
    {
        $response = $this->api->sendGetRequest( self::GET_PRICES );
        if ( $response ) {
            return $response;
        }

        return array( 'success' => false, 'list' => array() );
    }

    /**
     * Get list of all requests for SENDER IDs.
     *
     * @return array
     */
    public function getSenderIdsList()
    {
        $response = $this->api->sendGetRequest( self::GET_SENDER_IDS_LIST );
        if ( $response ) {
            $response['pending'] = null;
            foreach ( $response['list'] as &$item ) {
                $item['date'] = Utils\DateTime::formatDate( Utils\DateTime::UTCToWPTimeZone( $item['date'] ) );
                $item['status_date'] = $item['status_date'] ? Utils\DateTime::formatDate( Utils\DateTime::UTCToWPTimeZone( $item['status_date'] ) ) : '';
                switch ( $item['status'] ) {
                    case 0:
                        $item['status'] = __( 'Pending', 'bookly' );
                        $response['pending'] = $item['name'];
                        break;
                    case 1:
                        $item['status'] = __( 'Approved', 'bookly' );
                        break;
                    case 2:
                        $item['status'] = __( 'Declined', 'bookly' );
                        break;
                    case 3:
                        $item['status'] = __( 'Cancelled', 'bookly' );
                        break;
                }
            }

            return $response;
        }

        	return array( 'success' => false, 'list' => array(), 'pending' => null );
		}

    /**
     * Request new SENDER ID.
     *
     * @param string $sender_id
     * @return array|false
     */
    public function requestSenderId( $sender_id )
    {
        if ( $this->api->getToken() ) {
            $response = $this->api->sendPostRequest( self::REQUEST_SENDER_ID, array( 'name' => $sender_id ) );
            if ( $response ) {

                return $response;
            }
        }

        return false;
    }

    /**
     * Cancel request for SENDER ID.
     *
     * @return bool
     */
    public function cancelSenderId()
    {
        if ( $this->api->getToken() ) {
            $response = $this->api->sendGetRequest( self::CANCEL_SENDER_ID );
            if ( $response ) {

                return true;
            }
        }

        return false;
    }

    /**
     * Reset SENDER ID to default (Bookly).
     *
     * @return bool
     */
    public function resetSenderId()
    {
        if ( $this->api->getToken() ) {
            $response = $this->api->sendGetRequest( self::RESET_SENDER_ID );
            if ( $response ) {

                return true;
            }
        }

        return false;
    }

    /**
     * Get sender ID.
     *
     * @return string
     */
    public function getSenderId()
    {
        return $this->sender_id['value'];
    }

    /**
     * Get sender ID approval date.
     *
     * @return string
     */
    public function getSenderIdApprovalDate()
    {
        return $this->sender_id['approved_at'];
    }

    /**
     * Set number of undelivered sms.
     *
     * @param int $count
     */
    public function setUndeliveredSmsCount( $count )
    {
        update_option( 'bookly_sms_undelivered_count', (int) $count );
    }

    /**
     * Get number of undelivered sms.
     *
     * @return int
     */
    public static function getUndeliveredSmsCount()
    {
        return (int) get_option( 'bookly_sms_undelivered_count', 0 );
    }

    /**
     * @inheritDoc
     */
    public function translateError($error_code)
    {
        switch ( $error_code ) {
            case 'ERROR_INVALID_SENDER_ID': return __( 'Incorrect sender ID', 'bookly' );
            default: return null;
        }
    }

	    /**
     * @inheritDoc
     */
    protected function setupListeners()
    {
        $sms = $this;

        $this->api->listen( Events::ACCOUNT_PROFILE_LOADED, function ( $response ) use ( $sms ) {
            if ( isset( $response['account'][ Account::PRODUCT_SMS_NOTIFICATIONS ] ) ) {
                $sms->sender_id = $response['account'][ Account::PRODUCT_SMS_NOTIFICATIONS ]['sender_id'];
                $sms->setUndeliveredSmsCount( $response['account'][ Account::PRODUCT_SMS_NOTIFICATIONS ]['undelivered_count'] );
            } else {
                $sms->setUndeliveredSmsCount( 0 );
            }
        } );

        $this->api->listen( Events::ACCOUNT_PROFILE_NOT_LOADED, function () use ( $sms ) {
            $sms->setUndeliveredSmsCount( 0 );
        } );

        $this->api->listen( Events::ACCOUNT_LOGGED_OUT, function () use ( $sms ) {
            $sms->setUndeliveredSmsCount( 0 );
        } );
    }
}

